/**
  ************************************* Copyright ****************************** 
  *
  *                 Copyright (C), 2012-2016, Radioway Tech. Co., Ltd.
  *                            All Rights Reserved
  *
  *    
  * FileName   : board.c
  * Version    : v1.0
  * Author     : Heb
  * Date       : 2024-09-25
  * Description: 
  * Function List: 
  	1. ....
          <version>: 
     <modify staff>: 
             <data>: 
      <description>: 
  	2. ...
  ******************************************************************************
 */

#include "board.h"
#include <math.h>

#define TAG  "board"
#include "trace.h"
#include "utilities.h"

#define PORT_FLASH_BUFF_SIZE	128

static uint8_t  fac_us = 0;							//us延时倍乘数
//static uint16_t fac_ms = 0;							//ms延时倍乘数,在ucos下,代表每个节拍的ms数

uint32_t system_tick = 0;
volatile uint32_t TIM14_UpdateFlag = 0;
volatile unsigned char Tim14_Tick = 0;

BLINK blink;


void board_init()
{

}





void delay_init(void)
{
    fac_us = SystemCoreClock / 1000000;
//    fac_ms = SystemCoreClock / 1000;

    if (SysTick_Config(SystemCoreClock / 1000))
    {
        while (1)
        {
        }
    }

    NVIC_SetPriority(SysTick_IRQn, 0x0);
}

void delay_us(uint32_t nus)
{
    uint32_t ticks;
    uint32_t told, tnow, tcnt = 0;
    uint32_t reload = SysTick->LOAD;				//LOAD的值
    ticks = nus * fac_us; 						//需要的节拍数
    told = SysTick->VAL;        				//刚进入时的计数器值

    while(1)
    {
        tnow = SysTick->VAL;

        if(tnow != told)
        {
            if(tnow < told)
                tcnt += told - tnow;	//这里注意一下SYSTICK是一个递减的计数器就可以了.
            else 
                tcnt += reload - tnow + told;

            told = tnow;

            if(tcnt >= ticks)break;			//时间超过/等于要延迟的时间,则退出.
        }
    }
}

void delay_ms(uint32_t nms)
{
    uint32_t i;

    for(i = 0; i < nms; i++)
        delay_us(1000);
}


uint32_t hal_get_tick(void)
{
    return system_tick;
}

uint32_t hal_get_elapsed_time( uint32_t past )
{
    if ( past == 0 )
    {
        return 0;
    }

    uint32_t now = hal_get_tick( );

    // Intentional wrap around. Works Ok if tick duration below 1ms
    return now - past;
}

void hal_inc_tick(void)
{
    system_tick ++;
}


void SysTick_Handler(void)
{
    hal_inc_tick();
//    system_tick ++;
}

void IWDG_Configure(uint32_t Timeout)
{
    uint16_t Reload = LSI_VALUE / 1000 * Timeout / 32;

    /* Enable LSI Clock */
    SET_BIT(RCC->CSR, RCC_CSR_LSION);

    /* Wait LSI Ready */
    while (READ_BIT(RCC->CSR, RCC_CSR_LSIRDY) == 0)
    {
    }

    /* Wait Watchdog Prescaler Value Update Completed */
    while(READ_BIT(IWDG->SR, IWDG_SR_PVU));
    /* Disable The Protection And Enable Access To Other Configuration Registers */
    WRITE_REG(IWDG->KR, 0x5555);
    /* IWDG Prescaler Divider */
    MODIFY_REG(IWDG->PR, IWDG_PR_PR, 0x03U << IWDG_PR_PR_Pos);
    /* Wait Watchdog Prescaler Value Update Completed */
    while(READ_BIT(IWDG->SR, IWDG_SR_PVU));

    /* Wait Watchdog Counter Reload Value Update Completed */
    while(READ_BIT(IWDG->SR, IWDG_SR_RVU));
    /* Disable The Protection And Enable Access To Other Configuration Registers */
    WRITE_REG(IWDG->KR, 0x5555);
    /* Watchdog Counter Reload Value */
    MODIFY_REG(IWDG->RLR, IWDG_RLR_RL, Reload << IWDG_RLR_RL_Pos);
    /* Wait Watchdog Counter Reload Value Update Completed */
    while(READ_BIT(IWDG->SR, IWDG_SR_PVU));

    /* Feed The Dog */
    WRITE_REG(IWDG->KR, 0xAAAA);

    /* Start The Watchdog */
    WRITE_REG(IWDG->KR, 0xCCCC);
}

void TIM14_Configure(void)
{
    /* Enable TIM14 Clock */
    SET_BIT(RCC->APB1ENR, RCC_APB1ENR_TIM14);

    /* Config TIM14 Clock Division */
    MODIFY_REG(TIM14->CR1, TIM_CR1_CKD, 0x00U << TIM_CR1_CKD_Pos);

    /* Config TIM14 Edge Alignment Mode */
    MODIFY_REG(TIM14->CR1, TIM_CR1_CMS, 0x00U << TIM_CR1_CMS_Pos);

    /* Config TIM14 Up Count Direction */
    MODIFY_REG(TIM14->CR1, TIM_CR1_DIR, 0x00U << TIM_CR1_DIR_Pos);

    /* Config TIM14 Auto-Reload Value */
    WRITE_REG(TIM14->ARR, 10 - 1);

    /* Config TIM14 Prescaler Value */
//    WRITE_REG(TIM14->PSC, TIM14_GetFrequency() / 10000 - 1);
		WRITE_REG(TIM14->PSC, SystemCoreClock / 10000 - 1);

    /* Enable Update Interrupt */
    SET_BIT(TIM14->DIER, TIM_DIER_UIE);

    NVIC_SetPriority(TIM14_IRQn, 1);
    NVIC_EnableIRQ(TIM14_IRQn);

    /* Enable TIM14 */
    SET_BIT(TIM14->CR1, TIM_CR1_CEN);
}

/***********************************************************************************************************************
  * @brief  This function handles TIM14 Handler
  * @note   none
  * @param  none
  * @retval none
  *********************************************************************************************************************/
extern void lbn_timer_1ms_process(void);
extern uint32_t timestamp;
extern uint16_t sync_100ms_tick;
extern uint8_t time_print_flag;
extern uint16_t light_mode_timebase;
extern uint16_t sync_rand_count;
extern unsigned char sync_adv_flag;
void TIM14_IRQHandler(void)
{
    if (READ_BIT(TIM14->SR, TIM_SR_UIF) && READ_BIT(TIM14->DIER, TIM_DIER_UIE))
    {
        TIM14_UpdateFlag = 1;
		

        /* Clear TIM14 Update Interrupt Flag */
        CLEAR_BIT(TIM14->SR, TIM_SR_UIF);
    }
}


void port_read_flash(unsigned int addr,unsigned char *buf,unsigned int size)
{
	if ((buf == NULL) || (size == 0))
    {
        return;
    }

	addr = addr + FLASH_BASE_ADDR;
	
    for (uint32_t i = 0; i < size; i++)
    {
        buf[i] = *(uint8_t*) addr;
        addr ++;
    }
}

/****************************************************************************
@fn void port_write_flash(unsigned int  addr, unsigned char *buf,unsigned char size);     	  
@brief   
1、写FLASH
2、CPU时钟需要配置为1-16MHz之间的正整数，且Flash擦写之前需要配置FREQ_CLK寄存器，
该寄存器值即为当前CPU时钟频率。
@param[in]  
@return    
*****************************************************************************/ 
void port_write_flash(uint32_t addr, void *buf, uint16_t size )
{
    uint8_t* pdata = (uint8_t*)buf;

    BOARD_CRITICAL_BEGIN( );

	addr = addr + FLASH_BASE_ADDR;
	
    /* Unlock The Flash Memory */
    if (READ_BIT(FLASH->CR, FLASH_CR_LOCK))
    {
        WRITE_REG(FLASH->KEYR, 0x45670123);
        WRITE_REG(FLASH->KEYR, 0xCDEF89AB);
    }

    // Write data to Flash memory
    FLASH->CR |= FLASH_CR_PG;

    for (uint32_t i = 0; i < size; i+=2)
    {
        *(__IO uint8_t*)(addr + i) = pdata[i];
        
        if ((i+1)<size)
        {
            *(__IO uint8_t*)(addr + i + 1) = pdata[i +1];
        }
        
        while ((FLASH->SR & FLASH_SR_BSY) != 0);

        if ((*(__IO uint8_t*)(addr + i)) != pdata[i])
        {
            // Error occurred during write operation
            break;
        }
    }

    FLASH->CR &= ~FLASH_CR_PG;

    /* Lock The Flash Memory */
    SET_BIT(FLASH->CR, FLASH_CR_LOCK);
	
    BOARD_CRITICAL_END( );
}

void port_erase_flash(unsigned int addr)
{
	uint32_t secpos  = 0;	   //扇区地址

    BOARD_CRITICAL_BEGIN( );
	/* Unlock The Flash Memory */
    if (READ_BIT(FLASH->CR, FLASH_CR_LOCK))
    {
        WRITE_REG(FLASH->KEYR, 0x45670123);
        WRITE_REG(FLASH->KEYR, 0xCDEF89AB);
    }

    secpos = addr / FLASH_SECTOR_SIZE;	  //扇区地址  0~16

	// Erase Flash page
    FLASH->CR |= FLASH_CR_PER;
    FLASH->AR = secpos * FLASH_SECTOR_SIZE + FLASH_BASE_ADDR;
    FLASH->CR |= FLASH_CR_STRT;

    while ((FLASH->SR & FLASH_SR_BSY) != 0);

    FLASH->CR &= ~FLASH_CR_PER;
	
    /* Lock The Flash Memory */
    SET_BIT(FLASH->CR, FLASH_CR_LOCK);
	
    BOARD_CRITICAL_END( );
}
void port_init_flash(void)
{	
    u8  flash_data[PORT_FLASH_BUFF_SIZE];
    u8  i;
    u16 sec_addr = 0;	   //扇区地址
    port_read_flash(FLASH_BACKUP_FLAG_ADDR, (u8 *)&sec_addr, 2);
    
    if (sec_addr >= FLASH_DATA_BASE_SECTOR && (sec_addr <(FLASH_DATA_BASE_SECTOR + FLASH_DATA_SECTOR_SIZE)))
    {
        port_erase_flash(sec_addr);//erase old sector
	
        for(i = 0; i < (FLASH_SECTOR_SIZE / PORT_FLASH_BUFF_SIZE);i++)
        {
            port_read_flash(FLASH_BACKUP_ADDR + i * PORT_FLASH_BUFF_SIZE,flash_data,sizeof(flash_data));

            port_write_flash(sec_addr + i * PORT_FLASH_BUFF_SIZE,flash_data,sizeof(flash_data));
        }		
    
        // 清标记
        port_erase_flash(FLASH_BACKUP_FLAG_ADDR);//erase old sector
    }
}

/****************************************************************************
@fn  void port_modify_flash(unsigned int addr, unsigned char *buf,unsigned char size);     	  
@brief   
1、修改flash某个地址的内容，先拷贝数据到备份区域，再将数据写回原来的扇区。
2、CPU时钟需要配置为1-16MHz之间的正整数，且Flash擦写之前需要配置FREQ_CLK寄存器，
该寄存器值即为当前CPU时钟频率。
@param[in]  
@return    
*****************************************************************************/ 
void port_modify_flash(unsigned int addr, unsigned char *buf,unsigned int size)
{
	u8  flash_data[PORT_FLASH_BUFF_SIZE] = {0};
    
    u8 sub_index = 0;
	u8 sub_offset = 0;
    u8 sub_remain = 0;
    
    u8  i = 0,j = 0;
	u16 sec_addr = 0;
	
    if((addr % FLASH_SECTOR_SIZE+size)>FLASH_SECTOR_SIZE)
    {
        return;
    }
    
	sec_addr  = (addr / FLASH_SECTOR_SIZE) * FLASH_SECTOR_SIZE;
	
    sub_index = (addr % FLASH_SECTOR_SIZE)/PORT_FLASH_BUFF_SIZE;
    sub_offset = (addr % FLASH_SECTOR_SIZE)%PORT_FLASH_BUFF_SIZE;
    sub_remain = PORT_FLASH_BUFF_SIZE-sub_offset;
    
    if (size<=sub_remain)
        sub_remain = size;

	port_erase_flash(FLASH_BACKUP_ADDR);//erase backup sector
	for(i = 0; i < (FLASH_SECTOR_SIZE / PORT_FLASH_BUFF_SIZE);i++)
	{
		port_read_flash(sec_addr + i * PORT_FLASH_BUFF_SIZE,flash_data,sizeof(flash_data));

        if (sub_index == i)
        {
            for(j=0;j<sub_remain;j++)	   		//复制
            {
                flash_data[j+sub_offset]=buf[j];	  
            }
        }
        
		port_write_flash(FLASH_BACKUP_ADDR + i * PORT_FLASH_BUFF_SIZE,flash_data,sizeof(flash_data));
        
        if(size!=sub_remain && sub_index == i)
        {
            sub_index ++;
            sub_offset = 0;
            buf+=sub_remain;  				//指针偏移
            size -= sub_remain;
            if(size>PORT_FLASH_BUFF_SIZE)
                sub_remain=PORT_FLASH_BUFF_SIZE;           //下一个扇区还是写不完
			else 
                sub_remain=size;		//下一个扇区可以写完了
        }
	}
    
    // 做标记
    port_erase_flash(FLASH_BACKUP_FLAG_ADDR);//erase old sector
    port_write_flash(FLASH_BACKUP_FLAG_ADDR, (u8 *)&sec_addr, 2);
    
	port_erase_flash(sec_addr);//erase old sector
	for(i = 0; i < (FLASH_SECTOR_SIZE / PORT_FLASH_BUFF_SIZE);i++)
	{
		port_read_flash(FLASH_BACKUP_ADDR + i * PORT_FLASH_BUFF_SIZE, flash_data, sizeof(flash_data));

//        if (sub_index == i)
//        {
//            for(j=0;j<sub_remain;j++)	   		//复制
//            {
//                flash_data[j+sub_offset]=buf[j];	  
//            }
//        }
		port_write_flash(sec_addr + i * PORT_FLASH_BUFF_SIZE,flash_data,sizeof(flash_data));
        
//        if(size!=sub_remain && sub_index == i)
//        {
//            sub_index ++;
//            sub_offset = 0;
//            buf+=sub_remain;  				//指针偏移
//            size -= sub_remain;
//            if(size>PORT_FLASH_BUFF_SIZE)
//                sub_remain=PORT_FLASH_BUFF_SIZE;           //下一个扇区还是写不完
//			else 
//                sub_remain=size;		//下一个扇区可以写完了
//        }
	}	

     // 清标记
    port_erase_flash(FLASH_BACKUP_FLAG_ADDR);//erase old sector
}

void port_fill_flash(unsigned int fill_addr, unsigned char fill_data, unsigned int data_size)
{
	u8  flash_data[PORT_FLASH_BUFF_SIZE] = {0};
    
    u8 sub_index = 0;
	u8 sub_offset = 0;
    u8 sub_remain = 0;
    
    u8  i = 0,j = 0;
	u16 sec_addr = 0;
	
    if((fill_addr % FLASH_SECTOR_SIZE+data_size)>FLASH_SECTOR_SIZE)
        return;
    
	sec_addr  = (fill_addr / FLASH_SECTOR_SIZE) * FLASH_SECTOR_SIZE;
	
    sub_index = (fill_addr % FLASH_SECTOR_SIZE)/PORT_FLASH_BUFF_SIZE;
    sub_offset = (fill_addr % FLASH_SECTOR_SIZE)%PORT_FLASH_BUFF_SIZE;
    sub_remain = PORT_FLASH_BUFF_SIZE-sub_offset;
    
    if (data_size<=sub_remain)
        sub_remain = data_size;

	port_erase_flash(FLASH_BACKUP_ADDR);//erase backup sector
	for(i = 0; i < (FLASH_SECTOR_SIZE / PORT_FLASH_BUFF_SIZE);i++)
	{
		port_read_flash(sec_addr + i * PORT_FLASH_BUFF_SIZE,flash_data,sizeof(flash_data));

        if (sub_index == i)
        {
            for(j=0;j<sub_remain;j++)	   		//复制
            {
                flash_data[j+sub_offset]=fill_data;	  
            }
        }
        
		port_write_flash(FLASH_BACKUP_ADDR + i * PORT_FLASH_BUFF_SIZE,flash_data,sizeof(flash_data));
        
        if(data_size!=sub_remain && sub_index == i)
        {
            sub_index ++;
            sub_offset = 0;
            data_size -= sub_remain;
            if(data_size>PORT_FLASH_BUFF_SIZE)
                sub_remain=PORT_FLASH_BUFF_SIZE;           //下一个扇区还是写不完
			else 
                sub_remain=data_size;		//下一个扇区可以写完了
        }
	}
    
    // 做标记
    port_erase_flash(FLASH_BACKUP_FLAG_ADDR);//erase old sector
    port_write_flash(FLASH_BACKUP_FLAG_ADDR, (u8 *)&sec_addr, 2);
    
	port_erase_flash(sec_addr);//erase old sector
	
	for(i = 0; i < (FLASH_SECTOR_SIZE / PORT_FLASH_BUFF_SIZE);i++)
	{
		port_read_flash(FLASH_BACKUP_ADDR + i * PORT_FLASH_BUFF_SIZE,flash_data,sizeof(flash_data));

		port_write_flash(sec_addr + i * PORT_FLASH_BUFF_SIZE,flash_data,sizeof(flash_data));
	}		
    
    // 清标记
    port_erase_flash(FLASH_BACKUP_FLAG_ADDR);//erase old sector
}

u32 devinfo_get_net_address(unsigned char* addr)
{
    uint32_t crc_24;
    uint32_t temp_buf[3] = {0};

    temp_buf[0] = *(volatile uint32_t*)(0x1FFFF7E8);
    temp_buf[1] = *(volatile uint32_t*)(0x1FFFF7EC);
    temp_buf[2] = *(volatile uint32_t*)(0x1FFFF7F0);
	
	log_debug("UID: ", (uint8_t *)&temp_buf, 12);

    crc_24 = ll_crc24_generate((uint8_t *)temp_buf, sizeof(temp_buf));
	
	if(addr != NULL)
	{
		addr[0] = (unsigned char)crc_24;
		addr[1] = (unsigned char)(crc_24 >> 8);
		addr[2] = (unsigned char)(crc_24 >> 16);
	}
	
    return crc_24;
}


