/* ----------------------------------------------------------------------------
 * Copyright (c) 2020-2030 Boling Limited. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of Boling nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * -------------------------------------------------------------------------- */

/**
 * @file     main.c
 * @brief    BL1826 CORE1 main entry
 * @date     10. Sept. 2021
 * @author   Boling SW Team
 *
 * @version
 * Version 1.0
 *  - Initial release
 *
 * @{
 */

/*******************************************************************************
 * INCLUDES
 */
#include "om.h"
#include "om_driver.h"
#include "shell.h"
#include "evt.h"
#include "pm.h"
#include "board.h"

/*********************************************************************
 * MACROS
 */
#define TX_ROLE 1
#define OM24G_ACK_MODE 0
#define ENABLE_SLEEP_MODE 0
#if 0
//#undef OM_LOG_DEBUG
#undef OM_LOG_ARRAY_EX  
//#define OM_LOG_DEBUG(format, ...)
#define OM_LOG_ARRAY_EX(level, note, array, len)
#endif

/*******************************************************************************
 * TYPEDEFS
 */

/*********************************************************************
 * LOCAL VARIABLES
 */
static volatile uint16_t error_count = 0;
static volatile uint16_t right_count = 0;
static volatile uint16_t tx_count = 0;
static volatile uint16_t max_rty_count = 0;
static volatile uint16_t time_out_count = 0;

static uint8_t om24g_tx_payload[32];
/* 
In order to optimize reception performance, dual buffers are used to receive data, 
and the array size must be more than twice the maximum number of received packets, 
otherwise it will cause packet reception failure. For example, the business can only 
receive a maximum of 32 bytes of packets, and the array size is 64 bytes or greater
*/
static uint8_t om24g_rx_payload[64];

/**
 *******************************************************************************
 * @brief 24G configuration parameter
 *  
 * Package structure A、ACK mode、Dynamic length mode、Compatible with 6621C/D/E/F、nordic series
 * 5 bytes sync word : (sync_word0 + rx/tx_addr) 0xEDD4765602
 * 9 bits header: (a 6 bits’ length field, a 2-bit PID (Packet Identity) field and a 1 bit NO_ACK flag)
 * 0~32 bytes payload: {1,2,......32}
 * 2 bytes crc: seed = 0xFFFF crc_poly = 0x1021
 *
 * the data whitening scope includes only the payload domain.
 * the scope of CRC is fixed: includes sync_word field, 1-byte address field, 2 bytes guard field (if guard field exist), header field, and payload field.
 * 
 * @param[in] static_len  The maximum packet length is 32 bytes
 * @param[in] freq        2480MHZ
 *******************************************************************************
 */
om24g_config_t om24g_config = {
    .tx_data             = om24g_tx_payload,
    .rx_data             = om24g_rx_payload,
    .packet_struct_sel   = OM24G_STRUCTURE_A,
    .preamble            = 0xaa,
    .preamble_len        = 0x04,
    .sync_word_sel       = OM24G_SYNCWORD0,
    .sync_word0          = 0xEDD47656,
    .sync_word1          = 0x12345678,
    .tx_addr             = 0x02,
    .rx_addr             = 0x02,
    .addr_chk            = 0x01,
    .static_len          = 0x20,
    .hdr_bits            = 0x00,
    .addr1_bits          = 0x00,
    .len_bits            = 0x00,
    .addr1_pos           = 0x00,
    .len_pos             = 0x00,
    .endian              = OM24G_ENDIAN_MSB,
    .freq                = 2480,
    .data_rate           = OM24G_RATE_1M,
    .ack_en              = 1,
    .dpl_en              = 1,
    .white_en            = 1,
    .white_skip_hdr      = 1,
    .white_skip_addr     = 1,
    .white_skip_crc      = 1,
    .white_sel           = 0x00,
    .white_seed          = 0x80,
    .white_obit          = 0x07,
    .crc_len             = OM24G_CRC_2BYTE,
    .crc_en              = 1,
    .crc_mode            = 0,
    .crc_poly            = 0x1021, // 0x07,
    .crc_init            = 0xFFFF, // 0xFF,
    .crc_skip_sync       = 0,
    .crc_skip_len        = 0,
    .crc_skip_addr       = 0,
    .modulation_mode     = OM24G_MODULATION_GFSK,
    .detect_mode         = OM24G_SOFT_DETECTION,
};

/**
 *******************************************************************************
 * @brief 24G configuration parameter
 *  
 * Package structure A、NO_ACK mode、Fixed length mode、Compatible with 6621C/D/E/F，nordic series
 * 5 bytes sync word : (sync_word0 + rx/tx_addr) 0xEDD4765602
 * 0~32 bytes payload: {1,2,......32}
 * 2 bytes crc: seed = 0xFFFF crc_poly = 0x1021
 *
 * the data whitening scope includes only the payload domain.
 * the scope of CRC is fixed: includes sync_word field, 1-byte address field, 2 bytes guard field (if guard field exist),  and payload field.
 * 
 * @param[in] static_len  The package length is fixed at 32 bytes.
 * @param[in] freq        2480MHZ
 *******************************************************************************
 */
om24g_config_t om24g_config_noack = {
    .tx_data             = om24g_tx_payload,
    .rx_data             = om24g_rx_payload,
    .packet_struct_sel   = OM24G_STRUCTURE_A,
    .preamble            = 0xaa,
    .preamble_len        = 0x04,
    .sync_word_sel       = OM24G_SYNCWORD0,
    .sync_word0          = 0xEDD47656,
    .sync_word1          = 0xEDD47656,
    .tx_addr             = 0x02,
    .rx_addr             = 0x02,
    .addr_chk            = 0x01,
    .static_len          = 0x20,
    .hdr_bits            = 0x00,
    .addr1_bits          = 0x00,
    .len_bits            = 0x00,
    .addr1_pos           = 0x00,
    .len_pos             = 0x00,
    .endian              = OM24G_ENDIAN_MSB,
    .freq                = 2402,
    .data_rate           = OM24G_RATE_1M,
    .ack_en              = 0,
    .dpl_en              = 0,
    .white_en            = 1,
    .white_skip_hdr      = 1,
    .white_skip_addr     = 1,
    .white_skip_crc      = 1,
    .white_sel           = 0x00,
    .white_seed          = 0x80,
    .white_obit          = 0x07,
    .crc_len             = OM24G_CRC_2BYTE,
    .crc_en              = 1,
    .crc_mode            = 0,
    .crc_poly            = 0x1021, // 0x07,
    .crc_init            = 0xFFFF, // 0xFF,
    .crc_skip_sync       = 0,
    .crc_skip_len        = 0,
    .crc_skip_addr       = 0,
    .modulation_mode     = OM24G_MODULATION_GFSK,
    .detect_mode         = OM24G_SOFT_DETECTION,
};

/**
 *******************************************************************************
 * @brief 24G configuration parameter
 *  
 * Package structure B、NO ACK mode、Dynamic length mode、Compatible with BL1826、nordic、TI、siliconlab chip
 * 4 bytes sync word : 0xEDD47656
 * 1bytes header:      packet lenth
 * 0~32 bytes payload: {1,2,......32}
 * 2 bytes crc: seed = 0xFFFF crc_poly = 0x1021
 *
 * data whitening ranges include header field, payload field, and crc field. 
 * the scope of CRC includes sync_word, header, payload.
 * 
 * @param[in] static_len  The maximum packet length is 32 bytes
 * @param[in] freq        2480MHZ
 *******************************************************************************
 */
om24g_config_t om24g_config_b = {
    .tx_data             = om24g_tx_payload,
    .rx_data             = om24g_rx_payload,
    .packet_struct_sel   = OM24G_STRUCTURE_B,
    .preamble            = 0xaa,
    .preamble_len        = 0x01,
    .sync_word_sel       = OM24G_SYNCWORD0,
    .sync_word0          = 0xEDD47656,
    .sync_word1          = 0xEDD47656,
    .tx_addr             = 0x02,
    .rx_addr             = 0x02,
    .addr_chk            = 0x00,
    .static_len          = 0x20,
    .hdr_bits            = 0x08,
    .addr1_bits          = 0x00,
    .len_bits            = 0x08,
    .addr1_pos           = 0x00,
    .len_pos             = 0x00,
    .endian              = OM24G_ENDIAN_MSB,
    .freq                = 2406,
    .data_rate           = OM24G_RATE_1M,
    .ack_en              = 0,
    .dpl_en              = 1,
    .white_en            = 1,
    .white_skip_hdr      = 0,
    .white_skip_addr     = 1,
    .white_skip_crc      = 0,
    .white_sel           = 0x00,
    .white_seed          = 0x80,
    .white_obit          = 0x07,
    .crc_len             = OM24G_CRC_2BYTE,
    .crc_en              = 1,
    .crc_mode            = 0,
    .crc_poly            = 0x1021, // 0x07,
    .crc_init            = 0xFFFF, // 0xFF,
    .crc_skip_sync       = 0,
    .crc_skip_len        = 0,
    .crc_skip_addr       = 0,
    .modulation_mode     = OM24G_MODULATION_FSK,
    .detect_mode         = OM24G_SOFT_DETECTION,
};

/*********************************************************************
 * LOCAL FUNCTIONS
 */

/**
 *******************************************************************************
 * @brief  drv gpio isr handler
 *
 * @param[in] om_reg  om reg
 * @param[in] event  event
 * @param[in] param0  param0
 * @param[in] param1  param1
 *******************************************************************************
 */
static void drv_gpio_isr_handler(void *om_reg, drv_event_t event, void *int_status, void *data)
{
    OM_LOG_DEBUG("gpio: 0x%08X\n", (uint32_t)int_status);
}

/**
 *******************************************************************************
 * @brief  drv pin wakeup isr handler
 *
 * @param[in] om_reg  om reg
 * @param[in] event  event
 * @param[in] int_status  int status
 * @param[in] data  data
 *******************************************************************************
 */
static void drv_pin_wakeup_isr_handler(void *om_reg, drv_event_t event, void *int_status, void *data)
{
    OM_LOG_DEBUG("pinwakeup: 0x%08X\n", (uint32_t)int_status);
}

/**
 *******************************************************************************
 * @brief  pm sleep notify handler
 *
 * @param[in] sleep_state  sleep state
 * @param[in] power_status  power status
 *******************************************************************************
 */
static void pm_sleep_notify_handler(pm_sleep_state_t sleep_state, pm_status_t power_status)
{
    switch(sleep_state)
    {
        case PM_SLEEP_ENTRY:
            //OM_LOG_DEBUG("ENTRY\r\n");
            break;

        case PM_SLEEP_LEAVE_TOP_HALF:
            break;

        case PM_SLEEP_LEAVE_BOTTOM_HALF:
           //OM_LOG_DEBUG("BOTTOM_HALF\r\n");
            break;

        default:
            break;
    }
}

/**
 *******************************************************************************
 * @brief  common init
 *******************************************************************************
 */
static void hardware_init(void)
{
    drv_gpio_register_isr_callback(OM_GPIO0, drv_gpio_isr_handler);

    drv_pmu_wakeup_pin_set(BITMASK(PAD_BUTTON_0)|BITMASK(PAD_BUTTON_1), PMU_PIN_WAKEUP_LOW_LEVEL);
    drv_pmu_wakeup_pin_register_callback(drv_pin_wakeup_isr_handler);

    pm_sleep_notify_user_callback_register(pm_sleep_notify_handler);
}

static void om24g_callback(void *om_reg, drv_event_t drv_event, void *buff, void *num)
{
    uint16_t payload_lenth = 0;
    bool error_flag = false;

    switch (drv_event) {
        case DRV_EVENT_COMMON_RECEIVE_COMPLETED:
            payload_lenth = (uint32_t)num;
            for (int i = 0x00; i < (payload_lenth - 1); i++) {
                if (*((uint8_t *)buff + i) != *((uint8_t *)buff + i + 1)) {
                    error_flag = true;                   
                }
            }
            if (error_flag) {
                error_count++;
                error_flag = false;
                OM_LOG_DEBUG_ARRAY_EX("err Pkt", buff, payload_lenth);
            } else {
                right_count++;
                OM_LOG_DEBUG_ARRAY_EX("Pkt", buff, payload_lenth);
            }
            OM_LOG_DEBUG("E:%d  R:%d \r\n", error_count, right_count);

            #if ENABLE_SLEEP_MODE
            OM24G_CE_LOW();
            pm_sleep_allow(PM_ID_24G);
            om24g_control(OM24G_CONTROL_CLK_DISABLE, NULL);
            drv_pmu_ana_enable(false, PMU_ANA_RF_24G);
            drv_pmu_timer_control(PMU_TIMER_TRIG_VAL1, PMU_TIMER_CONTROL_SET_TIMER_VAL, (void *)(drv_pmu_timer_cnt_get() + PMU_TIMER_MS2TICK(100))); // PMU_TIMER_US2TICK
            drv_pmu_timer_control(PMU_TIMER_TRIG_VAL1, PMU_TIMER_CONTROL_ENABLE, NULL);
            #endif

            break;
        case DRV_EVENT_COMMON_RX_OVERFLOW:   // max retry
            break;
        case DRV_EVENT_COMMON_TRANSMIT_COMPLETED:
            pm_sleep_allow(PM_ID_24G);
            om24g_control(OM24G_CONTROL_CLK_DISABLE, NULL);
            drv_pmu_ana_enable(false, PMU_ANA_RF_24G);
            drv_pmu_timer_control(PMU_TIMER_TRIG_VAL1, PMU_TIMER_CONTROL_SET_TIMER_VAL, (void *)(drv_pmu_timer_cnt_get() + PMU_TIMER_MS2TICK(130))); // PMU_TIMER_US2TICK
            drv_pmu_timer_control(PMU_TIMER_TRIG_VAL1, PMU_TIMER_CONTROL_ENABLE, NULL);
            tx_count++;
            OM_LOG_DEBUG("tx_cnt: %d\r\n", tx_count);
            if(tx_count == 1000) {
                tx_count = 0;
                drv_pmu_timer_control(PMU_TIMER_TRIG_VAL1, PMU_TIMER_CONTROL_DISABLE, NULL);
            }
            break;
        case DRV_EVENT_COMMON_GENERAL:
            break;
        default:
            OM_ASSERT(0);
            break;
    }
}

static void om24g_callback_rx_to_tx(void *om_reg, drv_event_t drv_event, void *buff, void *num)
{
    switch (drv_event) {
        case DRV_EVENT_COMMON_RECEIVE_COMPLETED:
            drv_gpio_toggle(OM_GPIO0, BITMASK(8));
            right_count++;
            // OM_LOG_DEBUG_ARRAY_EX("Pkt", buff, (uint32_t)num);
            // OM_LOG_DEBUG("R:%d \r\n", right_count);
            OM24G_CE_LOW();
            #if !TX_ROLE
            om24g_write_int(om24g_tx_payload, 32);
            #endif
            break;
        case DRV_EVENT_COMMON_RX_OVERFLOW:   // max retry
            break;
        case DRV_EVENT_COMMON_TRANSMIT_COMPLETED:
            drv_gpio_toggle(OM_GPIO0, BITMASK(10));
            #if TX_ROLE
            drv_pmu_timer_control(PMU_TIMER_TRIG_VAL1, PMU_TIMER_CONTROL_SET_TIMER_VAL, (void *)(drv_pmu_timer_cnt_get() + PMU_TIMER_MS2TICK(200)));
            drv_pmu_timer_control(PMU_TIMER_TRIG_VAL1, PMU_TIMER_CONTROL_ENABLE, NULL);
            #endif
            om24g_read_int(om24g_rx_payload, 32);
            tx_count++;
            //OM_LOG_DEBUG("tx_cnt: %d\r\n", tx_count);
            break;
        case DRV_EVENT_COMMON_GENERAL:
            break;
        default:
            OM_ASSERT(0);
            break;
    }
}

static void om24g_callback_ack_mode(void *om_reg, drv_event_t drv_event, void *buff, void *num)
{
    uint16_t payload_lenth = 0;
    bool error_flag = false;

    switch (drv_event) {
        case DRV_EVENT_COMMON_RECEIVE_COMPLETED:
            drv_gpio_toggle(OM_GPIO0, GPIO_MASK(8));
            if(TX_ROLE) {
                pm_sleep_allow(PM_ID_24G);
                om24g_control(OM24G_CONTROL_CLK_DISABLE, NULL);
                drv_pmu_ana_enable(false, PMU_ANA_RF_24G);
            } else {
                static uint8_t ack_num = 0;
                for (uint8_t i = 0; i < 32; i++) {
                    om24g_tx_payload[i] = i+1;
                }
                ack_num++;
                om24g_write_ack(ack_num);
                OM_LOG_DEBUG_ARRAY_EX("ack_pkt", om24g_tx_payload, ack_num);
                if(ack_num > 31) {
                    ack_num = 0;
                }
                #if ENABLE_SLEEP_MODE
                OM24G_CE_LOW();
                pm_sleep_allow(PM_ID_24G);
                om24g_control(OM24G_CONTROL_CLK_DISABLE, NULL);
                drv_pmu_ana_enable(false, PMU_ANA_RF_24G);
                drv_pmu_timer_control(PMU_TIMER_TRIG_VAL1, PMU_TIMER_CONTROL_SET_TIMER_VAL, (void *)(drv_pmu_timer_cnt_get() + PMU_TIMER_MS2TICK(900))); // PMU_TIMER_US2TICK
                drv_pmu_timer_control(PMU_TIMER_TRIG_VAL1, PMU_TIMER_CONTROL_ENABLE, NULL);
                #endif
            }
            payload_lenth = (uint32_t)num;
            for (int i = 0; i < payload_lenth; i++) {
                if ((1 + i) != *((uint8_t *)buff + i)) {
                    error_flag = true;                  
                }
            }
            if (error_flag) {
                error_count++;
                error_flag = false;
                OM_LOG_DEBUG_ARRAY_EX("err Pkt", buff, payload_lenth);
            } else {
                right_count++;
                OM_LOG_DEBUG_ARRAY_EX("Pkt", buff, payload_lenth);
            }
            OM_LOG_DEBUG("ER:%d RX:%d MAX_RT:%d\r\n", error_count, right_count, max_rty_count);
            break;
        case DRV_EVENT_COMMON_RX_OVERFLOW:   // max retry
            max_rty_count++;
            OM_LOG_DEBUG("Max_Retry\r\n");
            pm_sleep_allow(PM_ID_24G);
            om24g_control(OM24G_CONTROL_CLK_DISABLE, NULL);
            drv_pmu_ana_enable(false, PMU_ANA_RF_24G);
            drv_pmu_timer_control(PMU_TIMER_TRIG_VAL1, PMU_TIMER_CONTROL_SET_TIMER_VAL, (void *)(drv_pmu_timer_cnt_get() + PMU_TIMER_MS2TICK(500))); // PMU_TIMER_US2TICK
            drv_pmu_timer_control(PMU_TIMER_TRIG_VAL1, PMU_TIMER_CONTROL_ENABLE, NULL);
            break;
        case DRV_EVENT_COMMON_TRANSMIT_COMPLETED:
            #if TX_ROLE
            tx_count++;
            OM_LOG_DEBUG("tx_cnt: %d\r\n", tx_count);
            if(tx_count == 1000) {
                tx_count = 0;
                drv_pmu_timer_control(PMU_TIMER_TRIG_VAL1, PMU_TIMER_CONTROL_DISABLE, NULL);
            }
            #endif
            break;
        case DRV_EVENT_COMMON_GENERAL:
            break;
        default:
            OM_ASSERT(0);
            break;
    }
}

#if TX_ROLE
static void om24g_write_it(void)
{
    static uint8_t j = 0;
 
    j++;
    if(j == 250) {
        j = 0;
    }
    for (uint8_t i = 0; i < 32; i++) {
        om24g_tx_payload[i] = j;
    }
    
    om24g_write_int(om24g_tx_payload, 32);
}

static void om24g_write_it_ack_mode(void)
{
    static uint16_t j = 1;

    for (uint8_t i = 0; i < 32; i++) {
        om24g_tx_payload[i] = i + 1;
    }
    om24g_register_event_callback(om24g_callback_ack_mode);
    om24g_write_int(om24g_tx_payload, j);
    j++;
    if (j > 32) {
        j = 1;
    }
}

/*Packet structure A transmits packets in polling mode and can communicate with function om24g_read_structure_a(void). */
static void om24g_write_structure_a(void)
{
    uint16_t total_count = 1000;
    static uint8_t j = 0;
    bool om24g_state = false;
 
    om24g_init(&om24g_config_noack);
    NVIC_DisableIRQ(OM24G_RF_IRQn);
    om24g_control(OM24G_CONTROL_DUMP_RF_REGISTER, NULL);
    while (total_count--) {
        j++;
        if(j == 250) {
            j = 0;
        }
        for (uint8_t i = 0; i < 32; i++) {
            om24g_tx_payload[i] = j;
        }
        om24g_state = om24g_write(om24g_tx_payload, 32);
        if (om24g_state == true) {
            tx_count++;
            OM_LOG_DEBUG("tx_cnt: %d\r\n", tx_count);
        }
        DRV_DELAY_MS(50);
    }
}

static void om24g_tx_to_rx_poll_mode(void)
{
    uint16_t total_count = 1000;
    static uint8_t j = 0;
    bool om24g_state = false;
    uint16_t payload_len = 0;
 
    om24g_init(&om24g_config_b);
    NVIC_DisableIRQ(OM24G_RF_IRQn);
    om24g_control(OM24G_CONTROL_DUMP_RF_REGISTER, NULL);
    while (total_count--) {
        j++;
        if(j == 250) {
            j = 0;
        }
        for (uint8_t i = 0; i < 32; i++) {
            om24g_tx_payload[i] = j;
        }
        om24g_state = om24g_write(om24g_tx_payload, 32);
        if (om24g_state == true) {
            tx_count++;
            //OM_LOG_DEBUG("tx_cnt: %d\r\n", tx_count);
        }

        payload_len = om24g_read(om24g_rx_payload, 4000);
        if(!payload_len) {
            OM_LOG_DEBUG("rx timeout\r\n");
        }
        OM_LOG_DEBUG_ARRAY_EX("Pkt", om24g_rx_payload, payload_len);

        DRV_DELAY_MS(1000);


    }
}

static void om24g_tx_to_rx_int_mode(void)
{
    
    for (uint8_t i = 0; i < 32; i++) {
        om24g_tx_payload[i] = 2;
    }
    om24g_init(&om24g_config_b);
    om24g_control(OM24G_CONTROL_DUMP_RF_REGISTER, NULL);
    om24g_register_event_callback(om24g_callback_rx_to_tx);
    om24g_write_int(om24g_tx_payload, 32);
}

/*Packet structure A transmits packets in interrupt mode and can communicate with function om24g_read_int_structure_a(void). */
static void om24g_write_int_structure_a(void)
{
    om24g_init(&om24g_config_noack);
	om24g_register_event_callback(om24g_callback);
    om24g_control(OM24G_CONTROL_DUMP_RF_REGISTER, NULL);
    om24g_write_it();
}

/*Packet structure A/B transmits packets in interrupt mode at a rate of 500K and can communicate with function om24g_read_int_test_500k(void). */
void om24g_write_int_test_500k(void)
{
    om24g_config.data_rate = OM24G_RATE_500K;
    om24g_config_noack.data_rate = OM24G_RATE_500K;
    om24g_write_int_structure_a();
}

/*Packet structure A/B transmits packets in interrupt mode at a rate of 500K and can communicate with function om24g_read_int_test_100k(void). */
void om24g_write_int_test_100k(void)
{
    om24g_config.data_rate = OM24G_RATE_100K;
    om24g_config_noack.data_rate = OM24G_RATE_100K;
    om24g_write_int_structure_a();
}

/*Packet structure B transmits packets in interrupt mode and can communicate with function om24g_read_int_structure_b(void). */
static void om24g_write_int_structure_b(void)
{
    om24g_init(&om24g_config_b);
    om24g_register_event_callback(om24g_callback);
    om24g_control(OM24G_CONTROL_DUMP_RF_REGISTER, NULL);
    om24g_write_it();
}

/**
 *******************************************************************************
 * @brief Packet structure B transmits packets in interrupt mode and can communicate with function om24g_read_rxaddr_check_B(void). 
 *  
 * 5 bytes sync word : (sync_word0 + tx_addr) 0xEDD47656AA
 * 2bytes header:      1byte tx address + 1byte packet lenth
 * The receiving end will compare whether tx_addr and rx_addr are equal, and if they are consistent, the packet will be received.
 *******************************************************************************
 */
static void om24g_write_txaddr_check_b(void)
{
    om24g_config_b.data_rate = OM24G_RATE_500K;
    om24g_config_b.dpl_en = 1;
    om24g_config_b.tx_addr = 0xAA;
    om24g_config_b.rx_addr = 0xAA;
    om24g_config_b.addr_chk = 1;
    om24g_config_b.hdr_bits = 0x10;
    om24g_config_b.addr1_bits = 0x08;
    om24g_config_b.len_bits = 0x08;
    om24g_config_b.addr1_pos = 0x08;
    om24g_config_b.len_pos = 0x00;

    om24g_init(&om24g_config_b);
	om24g_register_event_callback(om24g_callback);
    om24g_control(OM24G_CONTROL_DUMP_RF_REGISTER, NULL);
    om24g_write_it();
}

/*Packet structure A transmits packets in polling ACK mode and can communicate with function om24g_read_ack_mode_structure_a(void). */
static void om24g_write_ack_mode_structure_a(void)
{
    uint8_t j = 0;
    uint16_t payload_len = 0;
    uint16_t total_count = 1000;
    bool om24g_state = false;
    bool error_flag = false;

    for (uint8_t i = 0; i < 32; i++) {
        om24g_tx_payload[i] = i + 1;
    }
    om24g_init(&om24g_config);
    NVIC_DisableIRQ(OM24G_RF_IRQn);
    om24g_control(OM24G_CONTROL_DUMP_RF_REGISTER, NULL);
    while (total_count--) {
        j++;
        if (j > 32) {
            j = 1;
        }
        om24g_state = om24g_write(om24g_tx_payload, j);
        if (om24g_state) {
            payload_len = om24g_read_ack();
            for (int i = 0; i < payload_len; i++) {
                if ((i + 1) != om24g_rx_payload[i]) {
                    error_flag = true;                  
                }
            }
            if (error_flag) {
                error_count++;
                error_flag = false;
                OM_LOG_DEBUG_ARRAY_EX("err Pkt", om24g_rx_payload, payload_len);
            } else {
                right_count++;
                OM_LOG_DEBUG_ARRAY_EX("Pkt", om24g_rx_payload, payload_len);
            }
        } else {
            max_rty_count++;
            OM_LOG_DEBUG("Max_Retry\r\n");
        }
        drv_dwt_delay_ms(60);
    }
    OM_LOG_DEBUG("E:%d ACK:%d MAX_RT:%d \r\n", error_count, right_count, max_rty_count);
}

/**
 *******************************************************************************
 * @brief TX sensitivity test
 *  
 * Package structure A、NO_ACK mode、Fixed length mode、Compatible with 6621C/D/E/F，nordic series
 * 5 bytes sync word : (sync_word0 + rx/tx_addr) 0x9843AF0B46
 * 32 bytes payload:   {1,2,......32}
 *
 * whitening and crc disabled.
 *******************************************************************************
 */
static void om24g_tx_sensitivity_test(void)
{
    bool om24g_state = false;

    om24g_config_noack.sync_word0 = 0x9843AF0B;
    om24g_config_noack.tx_addr = 0x46;
    om24g_config_noack.addr_chk = 1;
    om24g_config_noack.white_en = 0;
    om24g_config_noack.crc_en = 0;
    om24g_config_noack.static_len  = 0x20;
    om24g_config_noack.freq = 2445;

    om24g_init(&om24g_config_noack);
    REGW(&OM_24G->PRE_GUARD, MASK_1REG(OM24G_GUARD_EN, 0x01));

    for (uint8_t i = 0; i < 32; i++) {
        om24g_tx_payload[i] = i + 1;
    }
    NVIC_DisableIRQ(OM24G_RF_IRQn);
    om24g_control(OM24G_CONTROL_DUMP_RF_REGISTER, NULL);
    while (1) {
        for (uint8_t i = 0; i < 32; i++) {
            om24g_tx_payload[i] = i;
        }
        om24g_state = om24g_write(om24g_tx_payload, 32);
        if (om24g_state == true) {
            tx_count++;
            OM_LOG_DEBUG("tx_cnt: %d\r\n", tx_count);
        }
        DRV_DELAY_MS(500);
    }
}

/*Packet structure A transmits packets in interrupt ACK mode and can communicate with function om24g_read_ack_mode_structure_a(void). */
static void om24g_write_int_ack_mode_structure_a(void)
{
    om24g_init(&om24g_config);
    om24g_control(OM24G_CONTROL_DUMP_RF_REGISTER, NULL);
    om24g_write_it_ack_mode();
}

static void om24g_tx_test(om24g_transfer_t tx_case)
{
    switch(tx_case) {
        case OM24G_SENSITIVITY:
            om24g_tx_sensitivity_test();
            break;
        case OM24G_POLL_STRUCTURE_A:
            /*Polling mode transmission packet*/
            om24g_write_structure_a(); 
            break;
        case OM24G_INT_STRUCTURE_A:
            /*Interrupt mode transmission packet, Package structure A*/
            om24g_write_int_structure_a();
            break;
        case OM24G_INT_STRUCTURE_B:
            /*Interrupt mode transmission packet, Package structure B*/
            om24g_write_int_structure_b();
            break;
        case OM24G_ACK_MODE_STRUCTURE_A:
            /*ACK Polling mode transmission packet*/
            om24g_write_ack_mode_structure_a();
            break;
        case OM24G_INT_ACK_MODE_STRUCTURE_A:
            /*ACK Interrupt mode transmission packet*/
            om24g_write_int_ack_mode_structure_a();
            drv_pmu_timer_control(PMU_TIMER_TRIG_VAL1, PMU_TIMER_CONTROL_SET_TIMER_VAL, (void *)(drv_pmu_timer_cnt_get() + PMU_TIMER_MS2TICK(500)));
            drv_pmu_timer_control(PMU_TIMER_TRIG_VAL1, PMU_TIMER_CONTROL_ENABLE, NULL);
            break;
        case OM24G_TX_RX_SWITCH_POLL:
            om24g_tx_to_rx_poll_mode();
            break;
        case OM24G_TX_RX_SWITCH_INT:
            om24g_tx_to_rx_int_mode();
            break;
        case OM24G_RX_ADDR_CHECK:
            om24g_write_txaddr_check_b();
            break;
        default:
            break;
    }  
}

#else

/*Packet structure A receives packets in polling mode and can communicate with function om24g_write_structure_a(void). */
static void om24g_read_structure_a(void)
{
    uint16_t payload_len = 0;
    uint16_t error_count = 0;
    uint16_t right_count = 0;
    bool error_flag = false;

    om24g_init(&om24g_config_noack);
    NVIC_DisableIRQ(OM24G_RF_IRQn);
    om24g_control(OM24G_CONTROL_DUMP_RF_REGISTER, NULL);
    while (1) {
        payload_len = om24g_read(om24g_rx_payload, 8000);
        OM_LOG_DEBUG("len:%d\r\n", payload_len); 
        if(!payload_len) {
            OM_LOG_DEBUG("rx timeout\r\n");
            break;
        }
        for (int i = 0x00; i < (payload_len - 1); i++) {
            if (om24g_rx_payload[i] != om24g_rx_payload[i+1]) {
                error_flag = true;
            }
        }
        if (error_flag) {
            error_count++;
            error_flag = false;
            OM_LOG_DEBUG_ARRAY_EX("err Pkt", om24g_rx_payload, payload_len);
        } else {
            right_count++;
            OM_LOG_DEBUG_ARRAY_EX("Pkt", om24g_rx_payload, payload_len);
        }
        OM_LOG_DEBUG("E:%d  R:%d \r\n", error_count, right_count);
    }
}

static void om24g_rx_to_tx_poll_mode(void)
{
    uint16_t total_count = 1000;
    static uint8_t j = 0;
    bool om24g_state = false;
    uint16_t payload_len = 0;
 
    om24g_init(&om24g_config_b);
    NVIC_DisableIRQ(OM24G_RF_IRQn);
    om24g_control(OM24G_CONTROL_DUMP_RF_REGISTER, NULL);
    while (total_count--) {
        j++;
        if(j == 250) {
            j = 0;
        }
        for (uint8_t i = 0; i < 32; i++) {
            om24g_tx_payload[i] = j;
        }
        payload_len = om24g_read(om24g_rx_payload, DRV_MAX_DELAY);
        if(!payload_len) {
            OM_LOG_DEBUG("rx timeout\r\n");
        }
        OM_LOG_DEBUG_ARRAY_EX("Pkt", om24g_rx_payload, payload_len);
        om24g_state = om24g_write(om24g_tx_payload, 32);
        if (om24g_state == true) {
            tx_count++;
            OM_LOG_DEBUG("tx_cnt: %d\r\n", tx_count);
        }
    }
}

static void om24g_rx_to_tx_int_mode(void)
{
    
    for (uint8_t i = 0; i < 32; i++) {
        om24g_tx_payload[i] = 2;
    }
    om24g_init(&om24g_config_b);
    om24g_control(OM24G_CONTROL_DUMP_RF_REGISTER, NULL);
    om24g_register_event_callback(om24g_callback_rx_to_tx);
    om24g_read_int(om24g_rx_payload, 32);
}

/*Packet structure A receives packets in interrupt mode and can communicate with function om24g_write_int_structure_a(void). */
static void om24g_read_int_structure_a(void)
{
    om24g_init(&om24g_config_noack);
    om24g_control(OM24G_CONTROL_DUMP_RF_REGISTER, NULL);
    om24g_register_event_callback(om24g_callback);
    om24g_read_int(om24g_rx_payload, 32);
}

/*Packet structure A/B receives packets in interrupt mode at a rate of 500K and can communicate with function om24g_write_int_test_500k(void). */
void om24g_read_int_test_500k(void)
{
    om24g_config.data_rate = OM24G_RATE_500K;
    om24g_config_noack.data_rate = OM24G_RATE_500K;
    om24g_read_int_structure_a();
}



/*Packet structure A/B receives packets in interrupt mode at a rate of 500K and can communicate with function om24g_write_int_test_100k(void). */
void om24g_read_int_test_100k(void)
{
    om24g_config.data_rate = OM24G_RATE_100K;
    om24g_config_noack.data_rate = OM24G_RATE_100K;
    om24g_read_int_structure_a();
}

/*Packet structure B receives packets in interrupt mode and can communicate with function om24g_write_int_structure_b(void). */
static void om24g_read_int_structure_b(void)
{
    om24g_init(&om24g_config_b);
    om24g_control(OM24G_CONTROL_DUMP_RF_REGISTER, NULL);
    om24g_register_event_callback(om24g_callback);
    om24g_read_int(om24g_rx_payload, 32);
}

/**
 *******************************************************************************
 * @brief Packet structure B transmits packets in interrupt mode and can communicate with function om24g_write_txaddr_check_B(void). 
 *  
 * 5 bytes sync word : (sync_word0 + tx_addr) 0xEDD47656AA
 * 2bytes header:      1byte tx address + 1byte packet lenth
 * The receiving end will compare whether tx_addr and rx_addr are equal, and if they are consistent, the packet will be received.
 *******************************************************************************
 */
static void om24g_read_rxaddr_check_b(void)
{
    om24g_config_b.data_rate = OM24G_RATE_500K;
    om24g_config_b.dpl_en = 1;
    om24g_config_b.tx_addr = 0xAA;
    om24g_config_b.rx_addr = 0xAA;
    om24g_config_b.addr_chk = 1;
    om24g_config_b.hdr_bits = 0x10;
    om24g_config_b.addr1_bits = 0x08;
    om24g_config_b.len_bits = 0x08;
    om24g_config_b.addr1_pos = 0x08;
    om24g_config_b.len_pos = 0x00;

    om24g_init(&om24g_config_b);
	om24g_register_event_callback(om24g_callback);
    om24g_control(OM24G_CONTROL_DUMP_RF_REGISTER, NULL);
    om24g_read_int(om24g_rx_payload, 32);
}

static uint8_t bitcount(uint8_t n)
{
    uint8_t count = 0;
    while (n) {
        count++;
        n &= (n - 1);
    }
    return count;
}

/**
 *******************************************************************************
 * @brief Receiving sensitivity test
 *  
 * Package structure A、NO_ACK mode、Fixed length mode、Compatible with 6621C/D/E/F，nordic series
 * 5 bytes sync word : (sync_word0 + rx/tx_addr) 0x9843AF0B46
 * 32 bytes payload:   {1,2,......32}
 *
 * whitening and crc disabled.
 *******************************************************************************
 */
static void om24g_rx_sensitivity_test(void)
{
    uint32_t error_bit = 0;
    uint16_t error_count = 0;
    uint16_t right_count = 0;
    bool error_flag = false;
    uint16_t payload_len = 0;
    //uint64_t timeout = 0;

    om24g_config_noack.sync_word0 = 0x9843AF0B;
    om24g_config_noack.rx_addr = 0x46;
    om24g_config_noack.addr_chk = 1;
    om24g_config_noack.white_en = 0;
    om24g_config_noack.crc_en = 0;
    om24g_config_noack.static_len  = 0x20;
    om24g_config_noack.freq = 2406;

    om24g_init(&om24g_config_noack);
    REGW(&OM_24G->PRE_GUARD, MASK_1REG(OM24G_GUARD_EN, 0x01));
    for (uint8_t i = 0; i < 32; i++) {
        om24g_tx_payload[i] = i + 1;
    }
    NVIC_DisableIRQ(OM24G_RF_IRQn);
    om24g_control(OM24G_CONTROL_DUMP_RF_REGISTER, NULL);
    while (1) {
        payload_len = om24g_read(om24g_rx_payload, DRV_MAX_DELAY);
        if (payload_len) {
            for (uint8_t i = 0; i < payload_len; i++) {
                if ((i + 1) != om24g_rx_payload[i]) {
                    error_flag = true;
                    error_bit += bitcount((i+1) ^ om24g_rx_payload[i]);
                }
            }
            if (error_flag) {
                error_flag = false;
                error_count++;
                //OM_LOG_DEBUG_ARRAY_EX("err Pkt", om24g_rx_payload, payload_len);
            } else {
                right_count++;
                //OM_LOG_DEBUG_ARRAY_EX("Pkt", om24g_rx_payload, payload_len);
                OM_LOG_DEBUG("\nT:%d  E:%d  R:%d  BitError:%d\r\n", (error_count + right_count), error_count, right_count, error_bit);

            }
            //timeout = 0;
            memset(om24g_rx_payload, 0, sizeof(om24g_rx_payload));
            payload_len = 0;
        } else {
            OM_LOG_DEBUG("rx timeout\r\n");
            break;
        }
//        timeout++;
//        if (timeout > 9000000) {
//            break;
//        }
    }
    OM24G_CE_LOW();
    OM_LOG_DEBUG("\nT:%d  E:%d  R:%d  BitError:%d\r\n", (error_count + right_count), error_count, right_count, error_bit);
}

/*Packet structure A receives packets in polling ACK mode and can communicate with function om24g_write_ack_mode_structure_a(void). */
static void om24g_read_ack_mode_structure_a(void)
{
    uint8_t j = 0;
    uint16_t payload_len = 0;
    bool error_flag = false;

    for (uint8_t i = 0; i < 32; i++) {
        om24g_tx_payload[i] = i + 1;
    }
    om24g_init(&om24g_config);
    NVIC_DisableIRQ(OM24G_RF_IRQn);
    om24g_control(OM24G_CONTROL_DUMP_RF_REGISTER, NULL);
    while (1) {
        j++;
        payload_len = om24g_read(om24g_rx_payload, 8000);
        if(!payload_len) {
            OM_LOG_DEBUG("rx timeout\r\n");
            break;
        }
        if (j > 32) {
            j = 1;
        }
        om24g_write_ack(j);
        OM_LOG_DEBUG_ARRAY_EX("ack_pkt", om24g_tx_payload, j);
        for (int i = 0; i < payload_len; i++) {
            if ((i + 1) != om24g_rx_payload[i]) {
                error_flag = true;                  
            }
        }
        if (error_flag) {
            error_count++;
            error_flag = false;
            OM_LOG_DEBUG_ARRAY_EX("err Pkt", om24g_rx_payload, payload_len);
        } else {
            right_count++;
            OM_LOG_DEBUG_ARRAY_EX("Pkt", om24g_rx_payload, payload_len);
        }
    }
    OM_LOG_DEBUG("E:%d  R:%d \r\n", error_count, right_count);
}

/*Packet structure A receives packets in interrupt ACK mode and can communicate with function om24g_write_ack_mode_structure_a(void). */
static void om24g_read_int_ack_mode_structure_a(void)
{
    om24g_init(&om24g_config);
    om24g_control(OM24G_CONTROL_DUMP_RF_REGISTER, NULL);
    om24g_register_event_callback(om24g_callback_ack_mode);
    om24g_read_int(om24g_rx_payload, 32);
}

static void om24g_rx_test(om24g_transfer_t rx_case)
{
    switch(rx_case) {
        case OM24G_SENSITIVITY:
            om24g_rx_sensitivity_test();
            break;
        case OM24G_POLL_STRUCTURE_A:
            /*Polling mode receiving packet*/
            om24g_read_structure_a(); 
            break;
        case OM24G_INT_STRUCTURE_A:
            /*Interrupt mode receiving packet,  Package structure A*/
            om24g_read_int_structure_a();
            break;
        case OM24G_INT_STRUCTURE_B:
            /*Interrupt mode transmission packet, Package structure B*/
            om24g_read_int_structure_b();
            break;
        case OM24G_ACK_MODE_STRUCTURE_A:
            /*ACK Polling mode receiving packet*/
            om24g_read_ack_mode_structure_a();
            break;
        case OM24G_INT_ACK_MODE_STRUCTURE_A:
            /*ACK Interrupt mode receiving packet*/
            om24g_read_int_ack_mode_structure_a();
            break;
        case OM24G_TX_RX_SWITCH_POLL:
            om24g_rx_to_tx_poll_mode();
            break;
        case OM24G_TX_RX_SWITCH_INT:
            om24g_rx_to_tx_int_mode();
            break;
        case OM24G_RX_ADDR_CHECK:
            om24g_read_rxaddr_check_b();
            break;
        default:
            break;
    }  
}
#endif
static void timer1_callback(void *om_reg, drv_event_t drv_event, void *param0, void *param1)
{
    OM24G_CE_LOW();
    drv_pmu_ana_enable(true, PMU_ANA_RF_24G); // Enabling before RF work, disable after work completion, and continuing to enable can cause excessive power consumption.
    om24g_control(OM24G_CONTROL_CLK_ENABLE, NULL);
    pm_sleep_prevent(PM_ID_24G);
    #if TX_ROLE
    #if OM24G_ACK_MODE
        om24g_write_it_ack_mode();
        drv_pmu_timer_control(PMU_TIMER_TRIG_VAL1, PMU_TIMER_CONTROL_SET_TIMER_VAL, (void *)(drv_pmu_timer_cnt_get() + PMU_TIMER_MS2TICK(130)));
        drv_pmu_timer_control(PMU_TIMER_TRIG_VAL1, PMU_TIMER_CONTROL_ENABLE, NULL);
    #else
        om24g_write_it();
    #endif

    #else

    #if ENABLE_SLEEP_MODE
    #if OM24G_ACK_MODE
        om24g_register_event_callback(om24g_callback_ack_mode);
    #else
        om24g_register_event_callback(om24g_callback);
    #endif
    om24g_read_int(om24g_rx_payload, 32);
    #endif

    #endif
    //OM_LOG_DEBUG("tim \r\n");
}

/*******************************************************************************
 * PUBLIC FUNCTIONS
 */

int main(void)
{
    drv_wdt_init(0);
	__enable_irq();
    board_init();
    hardware_init();
    drv_rf_init();
    evt_init();
    shell_init(NULL);
    pm_sleep_allow(PM_ID_SHELL);
    pm_sleep_prevent(PM_ID_24G);
    //drv_rf_txrx_pin_enable(true, 1);
    //OM_SYS->MON = 0x1003;
    
    OM_ASSERT_WHILE(true, RTE_OM24G_BL182X);
    OM_ASSERT_WHILE(true, (sizeof(om24g_rx_payload) >= (2*om24g_config.static_len)));
    OM_ASSERT_WHILE(true, (sizeof(om24g_rx_payload) >= (2*om24g_config_noack.static_len)));
    OM_ASSERT_WHILE(true, (sizeof(om24g_rx_payload) >= (2*om24g_config_b.static_len)));
    pm_sleep_enable(false);
    
    drv_rf_tx_power_set(false, RF_TX_POWER_5DBM);
    
    drv_rf_carrier_enable(true, 2445, 0);
    while(1);
    OM_LOG_DEBUG("start run\n");
    drv_pmu_timer_register_isr_callback(PMU_TIMER_TRIG_VAL1, timer1_callback);
#if TX_ROLE
    om24g_tx_test(OM24G_SENSITIVITY);
#else
    om24g_rx_test(OM24G_INT_STRUCTURE_B);

#endif
    // Main loop
    while (1) {
        // do event schedule
        evt_schedule();

        OM_CRITICAL_BEGIN();

        // if no event, do power manage
        if (evt_get_all() == 0) {
            pm_power_manage();
        }

        OM_CRITICAL_END();
    }
}
/** @} */
