/* ----------------------------------------------------------------------------
 * Copyright (c) 2020-2030 Boling Limited. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of Boling nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * -------------------------------------------------------------------------- */

/**
 * @file     main.c
 * @brief    mesh app entry
 * @date     01. Jan. 2023
 * @author   Boling SW Team
 *
 * @version
 * Version 1.0
 *  - Initial release
 *
 * @{
 */


/*******************************************************************************
 * INCLUDES
 */
#include "omsh_app.h"


/*********************************************************************
 * LOCAL FUNCTIONS
 */
static void hal_init(void)
{
    // Disable watchdog
    drv_wdt_init(0);

    // Board initialization
    board_init();

    // RF initialization
    drv_rf_init();

    // Event initialization
    evt_init();

    // Event timer initialization
    evt_timer_init();

    // Shell initialization
    #if (CONFIG_SHELL & APP_MESH_PTS)
    shell_init(omesh_shell_table);
    #else
    shell_init(NULL);
    #endif

    // Nvds initialization
    nvds_init(0);

    // Enable interrupts
    __enable_irq();
}

static void ble_stack_config(void)
{
    struct ob_stack_param param = {
        .max_connection         = OB_LE_HOST_CONNECTION_NB,
        .max_ext_adv_set        = OB_LE_HOST_ADV_SET_NUM,
        .max_att_mtu            = OB_LE_HOST_ATT_MTU,
        .max_gatt_serv_num      = OB_LE_HOST_MAX_GATT_SERV_NUM,
        .max_gatt_write_cache   = OB_LE_HOST_ATT_WRITE_CACHE_SIZE,
        .smp_sc_support         = OB_LE_HOST_SC_PAIRING,
    };
    omble_init(&param);
}

static void ble_mesh_config(void)
{
    // Enable mesh feature
    omesh_set_feature_en(APP_FEAT_RELAY,
        APP_FEAT_PROXY,
        APP_FEAT_FRND,
        APP_FEAT_LPN);

    // Init mesh app related
    omesh_app_init();

    // Init mesh stack
    omesh_stack_init();

    // Load mesh params
    omesh_storage_load();

    // Enable and start mesh
    omesh_stack_enable();
}


/*******************************************************************************
 * PUBLIC FUNCTIONS
 */
int main(void)
{
    // Init peripheral
    hal_init();

    // Config BLE environment
    ble_stack_config();

    // Config Mesh environment
    ble_mesh_config();

    // Main loop
    while (1) {
        // do event schedule
        evt_schedule();

        OM_CRITICAL_BEGIN();

        // if no event, do power manage
        if (evt_get_all() == 0) {
            pm_power_manage();
        }

        OM_CRITICAL_END();
    }
}


/** @} */
