/* ----------------------------------------------------------------------------
 * Copyright (c) 2020-2030 Boling Limited. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of Boling nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * -------------------------------------------------------------------------- */

/**
 * @file     example_lp_timer.c
 * @brief    example for low power timer
 * @date     3 Feb 2023
 * @author   Boling SW Team
 *
 * @defgroup EXAMPLE_LP_TIMER LP TIMER
 * @ingroup  DRIVER_EXAMPLES
 * @brief    Example for using lp timer
 * @details
 * There are some example as follows: various modes of using lp timer: free running mode,
 * buffered mode, one shot mode, double mode
 *
 * @version
 * Version 1.0
 *  - Initial release
 *
 * @{
 */


/*******************************************************************************
 * INCLUDES
 */
#include "om_driver.h"


/*******************************************************************************
 * MACROS
 */
/// Test pad for lp timer out0
#define PAD_LP_TIM_OUT0         18
/// Test pad for lp timer out1
#define PAD_LP_TIM_OUT1         15


/*******************************************************************************
 * TYPEDEFS
 */


/*******************************************************************************
 * CONST & VARIABLES
 */


/*******************************************************************************
 * LOCAL FUNCTIONS
 */


/*******************************************************************************
 * PUBLIC FUNCTIONS
 */
/**
 *******************************************************************************
 * @brief example of continuous pulse generation using lp timer
 *
 *******************************************************************************
 */
void example_lp_tim_free_running(void)
{
    uint8_t i = 1;
    lp_tim_free_running_config_t    mode_cfg;
    lp_tim_out_config_t             out_cfg;

    mode_cfg.presclar       = 0;
    mode_cfg.top_en         = 1;
    mode_cfg.top_val        = 500;
    mode_cfg.compare_val0   = 0;
    mode_cfg.compare_val1   = 0;
    drv_lp_tim_free_running_init(OM_LP_TIM, &mode_cfg);

    pin_config_t pin_cfg_cnt [] = {
        {PAD_LP_TIM_OUT0,  {PINMUX_LP_TIMER_OUT0_CFG},  PMU_PIN_MODE_PP, PMU_PIN_DRIVER_CURRENT_NORMAL},
    };
    drv_pin_init(pin_cfg_cnt, sizeof(pin_cfg_cnt) / sizeof(pin_cfg_cnt[0]));
    out_cfg.pol         = LP_TIM_POL_IDLE_LOW;
    out_cfg.action      = LP_TIM_UFOA_PULSE;
    drv_lp_tim_outx_config(OM_LP_TIM, LP_TIM_CHAN_OUT0, &out_cfg);
	
    drv_lp_tim_control(OM_LP_TIM, LP_TIM_CONTROL_POWER_IN_SLEEP, &i);
    drv_pmu_32k_enable_in_deep_sleep(true);

    drv_lp_tim_control(OM_LP_TIM, LP_TIM_CONTROL_START, NULL);
}

/**
 *******************************************************************************
 * @brief example of generating (rep0+1) pulses using lp time
 *
 *******************************************************************************
 */
void example_lp_tim_oneshot(void)
{
    uint8_t i = 1;
    lp_tim_one_shot_config_t    mode_cfg;
    lp_tim_out_config_t         out_cfg;

    mode_cfg.presclar       = 0;
    mode_cfg.rep0_val       = 3;
    mode_cfg.top_en         = 1;
    mode_cfg.top_val        = 500;
    mode_cfg.compare_val0   = 0;
    mode_cfg.compare_val1   = 0;
    drv_lp_tim_one_shot_init(OM_LP_TIM, &mode_cfg);

    pin_config_t pin_cfg_cnt [] = {
        {PAD_LP_TIM_OUT0,  {PINMUX_LP_TIMER_OUT0_CFG},  PMU_PIN_MODE_PP, PMU_PIN_DRIVER_CURRENT_NORMAL},
    };
    drv_pin_init(pin_cfg_cnt, sizeof(pin_cfg_cnt) / sizeof(pin_cfg_cnt[0]));
    out_cfg.pol         = LP_TIM_POL_IDLE_LOW;
    out_cfg.action      = LP_TIM_UFOA_PULSE;
    drv_lp_tim_outx_config(OM_LP_TIM, LP_TIM_CHAN_OUT0, &out_cfg);

    drv_lp_tim_control(OM_LP_TIM, LP_TIM_CONTROL_POWER_IN_SLEEP, &i);
    drv_pmu_32k_enable_in_deep_sleep(true);
	
    drv_lp_tim_control(OM_LP_TIM, LP_TIM_CONTROL_START, NULL);
}

/**
 *******************************************************************************
 * @brief example of using lp timer to generate a specified number of pulses
 * with a specified period
 *
 *******************************************************************************
 */
void example_lp_tim_buffered(void)
{
    uint8_t i = 1;
    lp_tim_buffered_config_t    mode_cfg;
    lp_tim_out_config_t         out_cfg;

    mode_cfg.presclar       = 0;
    mode_cfg.rep0_val       = 3;
    mode_cfg.rep1_val       = 6;
    mode_cfg.top_en         = 1;
    mode_cfg.top_val        = 500;
    mode_cfg.buftop_en      = 1;
    mode_cfg.buftop_val     = 1000;
    mode_cfg.compare_val0   = 0;
    mode_cfg.compare_val1   = 0;
    drv_lp_tim_buffered_init(OM_LP_TIM, &mode_cfg);

    pin_config_t pin_cfg_cnt [] = {
        {PAD_LP_TIM_OUT0,  {PINMUX_LP_TIMER_OUT0_CFG},  PMU_PIN_MODE_PP, PMU_PIN_DRIVER_CURRENT_NORMAL},
    };
    drv_pin_init(pin_cfg_cnt, sizeof(pin_cfg_cnt) / sizeof(pin_cfg_cnt[0]));
    out_cfg.pol         = LP_TIM_POL_IDLE_LOW;
    out_cfg.action      = LP_TIM_UFOA_PULSE;
    drv_lp_tim_outx_config(OM_LP_TIM, LP_TIM_CHAN_OUT0, &out_cfg);

    drv_lp_tim_control(OM_LP_TIM, LP_TIM_CONTROL_POWER_IN_SLEEP, &i);
    drv_pmu_32k_enable_in_deep_sleep(true);
	
    drv_lp_tim_control(OM_LP_TIM, LP_TIM_CONTROL_START, NULL);
}

/**
 *******************************************************************************
 * @brief example of using lp timer to generate two-way signals
 *
 *******************************************************************************
 */
void example_lp_tim_double(void)
{
    uint8_t i = 1;
    lp_tim_double_config_t      mode_cfg;
    lp_tim_out_config_t         out_cfg;

    mode_cfg.presclar       = 0;
    mode_cfg.rep0_val       = 3;
    mode_cfg.rep1_val       = 5;
    mode_cfg.top_en         = 1;
    mode_cfg.top_val        = 500;
    mode_cfg.compare_val0   = 0;
    mode_cfg.compare_val1   = 0;
    drv_lp_tim_double_init(OM_LP_TIM, &mode_cfg);

    pin_config_t pin_cfg_cnt [] = {
        {PAD_LP_TIM_OUT0,  {PINMUX_LP_TIMER_OUT0_CFG},  PMU_PIN_MODE_PP, PMU_PIN_DRIVER_CURRENT_NORMAL},
        {PAD_LP_TIM_OUT1,  {PINMUX_LP_TIMER_OUT1_CFG},  PMU_PIN_MODE_PP, PMU_PIN_DRIVER_CURRENT_NORMAL},
    };
    drv_pin_init(pin_cfg_cnt, sizeof(pin_cfg_cnt) / sizeof(pin_cfg_cnt[0]));
    out_cfg.pol         = LP_TIM_POL_IDLE_LOW;
    out_cfg.action      = LP_TIM_UFOA_PULSE;
    drv_lp_tim_outx_config(OM_LP_TIM, LP_TIM_CHAN_OUT0, &out_cfg);
    drv_lp_tim_outx_config(OM_LP_TIM, LP_TIM_CHAN_OUT1, &out_cfg);

    drv_lp_tim_control(OM_LP_TIM, LP_TIM_CONTROL_POWER_IN_SLEEP, &i);
    drv_pmu_32k_enable_in_deep_sleep(true);
	
    drv_lp_tim_control(OM_LP_TIM, LP_TIM_CONTROL_START, NULL);
}


/** @} */